#  Software: linebreak.awk v1.0

#  Purpose: To break long lines in text into shorter according provided length parameters
#  Usage format: awk -f linebreak.awk [file_name] [start_after_position max_line_length]

#  Usage examples:
#     linebreak.awk help
#     linebreak.awk version
#     linebreak.awk a.txt 70 90
#     linebreak.awk 70 90

#  (c)2024, Vartiklis, Jonas Skendelis. The linebreak.pl is distributed under MIT License (see license.txt file)
#           allowing to use, copy, modify, merge, publish, distribute, sublicense, and/or sell its copies. 
#  Its copyright notice and license text shall be included in all copies or substantial portions of this Software.

BEGIN {
   ORS="\n" ; MINPOS = 68; MAXPOS = 80;

   VERSION = "linebreak.awk v1.0"
   DEFINITION = "To break long lines in text into shorter (v.1.0, 2024).";
   SHORT_HELP = "Format: linebreak.awk [file_name | -] [start_position max_line_length]";

   COPYRIGHT = "(c)2024, Vartiklis, Jonas Skendelis. The linebreak.awk is distributed under" ORS \
            "MIT License (see license.txt file) allowing to use, copy, modify, merge, publish," ORS \
            "distribute, sublicense, and/or sell its copies.";

FULL_HELP = "Formats: " ORS \
"awk -f linebreak.awk [file_name | -] start_position max_line_length" ORS \
"awk -f linebreak.awk help" ORS \
"awk -f linebreak.awk version" ORS \
ORS
"Notes: 1) default positions are " MINPOS " (start position) and " MAXPOS " (maximal length);" ORS \
"2) processing is limited to the one file only; use of multiple files are fotnided." ORS \
"3) if a start position is bigger than maximal length then bigger value is used for both." ORS \
ORS \
"Detailed description" ORS \
ORS \
"Two parameters are used to control splitting: one to define the minimal length" ORS \
"and another to define maximal length. A line part till minimal length is kept" ORS \
"as is and then further the space is searching to make the split in that position." ORS \
"If space is not found then break happens at maximal length position." ORS \
ORS \
"The command below reads a text from standard input (console) and breaks line" ORS \
"leaving at least of 70 symbols but not longer as 90 symbols. The break will" ORS \
"be done at first space after 70th position. The result will be to standard" ORS \
"output stream (console)." ORS \
ORS \
"awk -f linebreak.awk 70 90" ORS ORS \
"Note: as by awk features you can use - (minus) sign in place of file" ORS \
"(it means to use console instead of file)." ORS \
"So you can use for performing the same as in command above:" ORS \
"awk -f linebreak.awk - 70 90" ORS \ 
ORS \
"But it is possible to redirect a result into a file (by using >), for example," ORS \
"awk -f linebreak.awk 70 90 > splited.txt" ORS \
ORS \
"Input file could be taken from redirection into input stream (by using <), for example," ORS \
ORS \
"awk -f linebreak.awk 70 90 <text.txt" ORS \
"or transferring in pipe (by using | ), for example," ORS \
"more text.txt | awk -f linebreak.awk 70 90" ORS \
"or providing in first position of command, for example," ORS \
"awk -f linebreak.awk text.txt 70 90" ORS \
ORS
"But even more - you can provide start position and maximal line length by using" ORS \
"MIN and MAX variables rather than by script parameters. For example:" ORS \
"awk -v MIN=60 -v MAX=78 -f linebreak.awk text.txt" ORS \
ORS
"Note: script parameters overrides values transfered by variables." ORS \

   tested = testPositionParamValues(MIN, MAX);
   if ( tested > 0 ) {
        errorText = generateNotIntegerMessage( tested == 1 ? MIN : MAX ) 
        printError(errorText)
        exit
   }

  if ( ARGC == 2 ) {
     if (getline < ARGV[1] < 0) {
        if (tolower(ARGV[1]) == "help") {
            printFullHelp()
            exit
        }
        if (tolower(ARGV[1]) == "version") {
           printVersion()             
            exit
        }
     } 
  }

  if ( ARGC == 3 || ARGC == 4 ) {
     tested = testParameters(ARGV[ARGC-2], ARGV[ARGC-1]);
     if (tested > 0) {
        errorText = generateNotIntegerMessage(ARGV[ tested + ARGC - 3] ) 
        printError(errorText)
        exit
     }
     MINPOS = ARGV[ARGC-2]
     MAXPOS = ARGV[ARGC-1]
  } else {
     if ( ARGC > 4 ) {
        printHelp();
        exit
     }
  }

  if ( ARGC == 3 ) {
     ARGC = 1 
  } else {
     ARGC = 2
  }
} # end starting block

{ 
   tail = $0;
   while (length(tail) > MINPOS) { 
      cut_off = substr( tail, 1, MINPOS)
      slice = substr(tail, MINPOS+1, MAXPOS-MINPOS)
      # print splited ORS piece
      bpos = index(slice, " ")
      if (bpos == 0) {
 	 cut_off = cut_off slice;
         tail = substr(tail, MINPOS + length(slice)+1)
      } else {
          cut_off = cut_off substr(slice, 1, bpos)
	  tail = substr (slice, bpos+1) substr(tail, MAXPOS+1)
      }
      print cut_off
   }
   print tail
} 

# ---------------- Subroutines area -----------------------------

function testParameters( minLength, maxLength ) {
   if (minLength ~ /^[0-9]+$/) {
       if (! (maxLength ~ /^[0-9]+$/)) {
         return 2
      }
   } else {
     return 1
   }
   return 0
}

function testPositionParamValues ( minPosParam, maxPosParam )  {
   if ( minPosParam ) {
       MINPOS = minPosParam
   } else {
     minPosParam = MINPOS
   }
   if ( maxPosParam ) {
       MAXPOS = maxPosParam
   } else {
     maxPosParam = MAXPOS
   }

   if (MINPOS > MAXPOS) {
	MAXPOS = MINPOS
   }

  return testParameters(minPosParam, maxPosParam)   
 }

function printHelp() {
   print DEFINITION ORS ORS SHORT_HELP
}

function printFullHelp() {
   print DEFINITION ORS ORS FULL_HELP ORS ORS COPYRIGHT
}
function printVersion() {               
   print DEFINITION ORS ORS VERSION ORS ORS COPYRIGHT
}
function generateNotIntegerMessage( paramValue ) {
  return "Position value '" paramValue "' is not integer value."
}   
function printError( errText ) {
  print DEFINITION ORS ORS "ERROR: " errText ORS ORS COPYRIGHT
}
